/*
 * Modified at March 2014.
 * RIKEN, Japan
 */

/*
* Binary and Source Code License Agreement for jV Version 4.3.1
* Copyright (c) 2003-2013, Protein Data Bank Japan
* All rights reserved.
* 
* The software, jV, and the associated documentation files (the 
* "Software") were developed by Drs. Kengo Kinoshita and Haruki Nakamura 
* in Protein Data Bank Japan at Institute for Protein Research, Osaka 
* University (the "Contributors"), which is supported by grant-in-aid 
* from National Bioscience Database Center, Japan Science and Technology
* Agancy (JST-NBDC) and Osaka University.
* 
* Redistribution and use in source and binary forms, with or without
* modification, are permitted provided that the following conditions 
* are met:
* 
* *  Redistributions of the source code for the Software must retain the
*    above copyright notice, this list of conditions and the following
*    disclaimer.
* 
* *  Redistributions in the binary form for the Software must reproduce
*    the above copyright notice, this list of conditions and the following
*    disclaimer in the documentation and/or other materials provided with
*    the distribution.
* 
* *  Neither the name of the Institute for Protein Research, Osaka University
*    nor the names of the Contributors may be used to endorse or promote
*    products derived from this software without specific prior written
*    permission.
* 
*    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
*    "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
*    LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS 
*    FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE 
*    COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
*    INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, 
*    BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
*    LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER 
*    CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT 
*    LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN 
*    ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
*    POSSIBILITY OF SUCH DAMAGE.
*/
package org.pdbj.viewer.images;

import org.pdbj.viewer.utils.*;

/**
 * Image of molecules.
 */
public abstract class MoleculeImage
{
    public static final float DEFAULT_CARTOON_HEIGHT = 0.2f;

    private Matrix4f _transform;

    private boolean _hbondBackbone;
    private boolean _ssbondBackbone;
    private float   _cartoonHeight;
    private int _residueBackColorIndex;
    private boolean _cartoonLoopTube;
    private boolean _cartoonRound;

    /**
     * Returns the transform matrix.
     */
    public Matrix4f getTransform()
    {
        return _transform;
    }

    /**
     * Sets whether hbond images are drawed between backbones.
     */
    public void setHBondBackbone(boolean enable)
    {
        _hbondBackbone = enable;
    }
    
    /**
     * Tests whether hbond images are drawed between backbones.
     */
    public boolean hbondBackbone()
    {
        return _hbondBackbone;
    }

    /**
     * Sets whether ssbond images are drawed between backbones.
     */
    public void setSSBondBackbone(boolean enable)
    {
        _ssbondBackbone = enable;
    }
    
    /**
     * Tests whether ssbond images are drawed between backbones.
     */
    public boolean ssbondBackbone()
    {
        return _ssbondBackbone;
    }

    /**
     * Sets the cartoon height.
     */
    public void setCartoonHeight(float height)
    {
        _cartoonHeight = height;
    }

    /**
     * Returns the cartoon height.
     */
    public float getCartoonHeight()
    {
        return _cartoonHeight;
    }

    /**
     * Sets the residue back color index.
     */
    public void setResidueBackColorIndex(int residueBackColorIndex)
    {
        _residueBackColorIndex = residueBackColorIndex;
    }

    /**
     * Returns the residue back color index.
     */
    public int getResidueBackColorIndex()
    {
        return _residueBackColorIndex;
    }

    /**
     * Sets whether loops are drawed by tube model.
     */
    public void setCartoonLoopTube(boolean cartoonLoopTube)
    {
        _cartoonLoopTube = cartoonLoopTube;
    }

    /**
     * Returns whether loops are drawed by tube model.
     */
    public boolean getCartoonLoopTube()
    {
        return _cartoonLoopTube;
    }

    /**
     * Sets whether square pillar with round edge is drawed in the cartoon representation.
     */
    public void setCartoonRound(boolean cartoonRound)
    {
        _cartoonRound = cartoonRound;
    }

    /**
     * Returns whether square pillar with round edge is drawed in the cartoon representation.
     */
    public boolean getCartoonRound()
    {
        return _cartoonRound;
    }

    /**
     * Sets whether bond images have been initialized.
     */
    public abstract void setBondInitialized(boolean initialized);

    /**
     * Tests whether bond images have been initialized.
     */
    public abstract boolean bondInitialized();

    /**
     * Sets whether backbone images have been initialized.
     */
    public abstract void setBackboneInitialized(boolean initialized);
    
    /**
     * Tests whether backbone images have been initialized.
     */
    public abstract boolean backboneInitialized();

    /**
     * Sets whether hbond images have been initialized.
     */
    public abstract void setHBondInitialized(boolean initialized);
    
    /**
     * Tests whether hbond images have been initialized.
     */
    public abstract boolean hbondInitialized();

    /**
     * Sets whether ssbond images have been initialized.
     */
    public abstract void setSSBondInitialized(boolean initialized);
    
    /**
     * Tests whether ssbond images have been initialized.
     */
    public abstract boolean ssbondInitialized();

    /**
     * Sets whether residue images should be refreshed.
     */
    public abstract void setRefreshResidue(boolean refresh);

    /**
     * Tests whether residue images should be refreshed.
     */
    public abstract boolean refreshResidue();

    /**
     * Default constructor.
     */
    public MoleculeImage()
    {
        _transform     = Matrix4f.identity();
        _cartoonHeight = DEFAULT_CARTOON_HEIGHT;
        _residueBackColorIndex = -1;
        _cartoonLoopTube = true;
        _cartoonRound = true;
    }
}

